/**
 * \file sdc_perm.c
 *
 * \brief Functions etc. required by to manage permissions of
 * keys
 *
 * \author Christoph Gellner (cgellner@de.adit-jv.com)
 *
 * \copyright (c) 2015 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/

#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <stdio.h>
#include <string.h>

#include <sdc.h>
#include <sdc_perm.h>
#include <private/sdc_intern.h>

/* Definitions types and defaults */

static const sdc_permissions_t initial_sdc_permissions = {
    0,     // UID of root
    0,     // GID of root
    SDC_PERM_UID_DFLT,
    SDC_PERM_GID_DFLT,
    SDC_PERM_OTHERS_DFLT,
    SDC_PERM_INHERIT_UID_DFLT,
    SDC_PERM_INHERIT_GID_DFLT,
    SDC_PERM_INHERIT_OTHERS_DFLT
};



/* Functions */

sdc_error_t sdc_permissions_alloc (sdc_permissions_t **permissions)
{
    if (permissions == NULL) {
        return SDC_INVALID_PARAMETER;
    }

    *permissions = malloc(sizeof(sdc_permissions_t));
    if (*permissions == NULL) {
        return SDC_NO_MEM;
    }

    memcpy(*permissions, &initial_sdc_permissions, sizeof(sdc_permissions_t));

    return SDC_OK;
}

sdc_error_t sdc_permissions_free (sdc_permissions_t *permissions)
{
    if (permissions == NULL) {
        return SDC_INVALID_PARAMETER;
    }

    free(permissions);

    return SDC_OK;
}

sdc_error_t sdc_set_default_permissions_and_gid (sdc_permissions_t *permissions, gid_t gid)
{
    if (permissions == NULL) {
        return SDC_INVALID_PARAMETER;
    }

    memcpy(permissions, &initial_sdc_permissions, sizeof(sdc_permissions_t));

    permissions->uid = geteuid();
    permissions->gid = gid;

    return SDC_OK;
}

sdc_error_t sdc_set_default_permissions_current_gid (sdc_permissions_t *permissions)
{
    return sdc_set_default_permissions_and_gid (permissions, getegid());
}

sdc_error_t sdc_permissions_set_gid (sdc_permissions_t *permissions, gid_t gid)
{
    if (permissions == NULL) {
        return SDC_INVALID_PARAMETER;
    }

    permissions->gid = gid;

    return SDC_OK;
}

sdc_error_t sdc_permissions_set_uid (sdc_permissions_t *permissions, uid_t uid)
{
    if (permissions == NULL) {
        return SDC_INVALID_PARAMETER;
    }

    permissions->uid = uid;

    return SDC_OK;
}

sdc_error_t sdc_permissions_get_gid (const sdc_permissions_t *permissions, gid_t *gid)
{
    if (permissions == NULL) {
        return SDC_INVALID_PARAMETER;
    }

    if (gid == NULL) {
        return SDC_INVALID_PARAMETER;
    }

    *gid = permissions->gid;

    return SDC_OK;
}

sdc_error_t sdc_permissions_get_uid (const sdc_permissions_t *permissions, uid_t *uid)
{
    if (permissions == NULL) {
        return SDC_INVALID_PARAMETER;
    }

    if (uid == NULL) {
        return SDC_INVALID_PARAMETER;
    }

    *uid = permissions->uid;

    return SDC_OK;
}


static sdc_error_t _eval_substring(sdc_permissions_t *perms,
                                   char *str)
{
    sdc_perm_bmsk_t *targets[6] = {NULL, NULL, NULL};
    char *operator;
    char *ptr;
    sdc_perm_bmsk_t mask;
    int i;

    if (!str || !strlen(str))
        return SDC_OK;

    /* How to modify? */
    operator = strchr(str, '=');
    if (operator)
        goto operator_found;

    operator = strchr(str, '+');
    if (operator)
        goto operator_found;

    operator = strchr(str, '-');
    if (operator)
        goto operator_found;

    return SDC_INVALID_PARAMETER; /* No valid operator, give up */

operator_found:

    /* What to modify? */
    for (ptr = str; ptr != operator; ptr++) {
        switch(*ptr) {
        case 'u':
            targets[0] = &perms->perms_owner;
            break;
        case 'g':
            targets[1] = &perms->perms_group;
            break;
        case 'o':
            targets[2] = &perms->perms_others;
            break;
        case 'U':
            targets[3] = &perms->perms_inherit_owner;
            break;
        case 'G':
            targets[4] = &perms->perms_inherit_group;
            break;
        case 'O':
            targets[5] = &perms->perms_inherit_others;
            break;
        case 'a':
            targets[0] = &perms->perms_owner;
            targets[1] = &perms->perms_group;
            targets[2] = &perms->perms_others;
            break;
        case 'A':
            targets[3] = &perms->perms_inherit_owner;
            targets[4] = &perms->perms_inherit_group;
            targets[5] = &perms->perms_inherit_others;
            break;
        case ' ':
            break;
        default:
            return SDC_INVALID_PARAMETER;
        }
    }

    /* Find out the bitmask */
    mask = 0;
    for (ptr = operator + 1; *ptr; ptr++) {
        switch(*ptr) {
        case 'e':
            mask |= SDC_PERM_ENCRYPT;
            break;
        case 'd':
            mask |= SDC_PERM_DECRYPT;
            break;
        case 's':
            mask |= SDC_PERM_SIGN;
            break;
        case 'v':
            mask |= SDC_PERM_VERIFY;
            break;
        case 'w':
            mask |= SDC_PERM_WRAP;
            break;
        case 'u':
            mask |= SDC_PERM_UNWRAP;
            break;
        case 'k':
            mask |= SDC_PERM_WRAP_KEY;
            break;
        case 'K':
            mask |= SDC_PERM_UNWRAP_KEY;
            break;
        case 'D':
            mask |= SDC_PERM_DELETE;
            break;
        case ' ':
            break;
        default:
            return SDC_INVALID_PARAMETER;
        }
    }

    /* Apply the operation to the target masks */
    switch(*operator) {
    case '=':
        for (i = 0; i < 6; i++)
            if (targets[i])
                *targets[i] = mask;
        break;
    case '+':
        for (i = 0; i < 6; i++)
            if (targets[i])
                *targets[i] |= mask;
        break;
    case '-':
        for (i = 0; i < 6; i++)
            if (targets[i])
                *targets[i] &= ~mask;
        break;
    default:
        /* Not possible */
        break;
    }
    return SDC_OK;
}

sdc_error_t sdc_permissions_apply_string(sdc_permissions_t *perms,
                                         const char *str, size_t maxlen)
{
    sdc_error_t res;
    char *buf;
    char *subs;
    size_t bsize;

    if (!str || !perms) {
        res = SDC_INVALID_PARAMETER;
        goto out;
    }

    bsize = strlen(str) + 1;
    if (bsize > (maxlen + 1))
        bsize = maxlen + 1;

    buf = malloc(bsize);
    if (!buf) {
        res = SDC_NO_MEM;
        goto out;
    }

    strncpy(buf, str, bsize);
    buf[bsize - 1] = 0;

    for (subs = strtok(buf, ","); subs; subs = strtok(NULL, ",")) {
        res = _eval_substring(perms, subs);
        if (res != SDC_OK)
            break;
    }
    free(buf);
out:
    return res;
}

static void _write_perms(sdc_perm_bmsk_t perm, char *target, char *space)
{
    sprintf(target,"%c%c%c%c%c%c%c%c%c%s",
            (perm & SDC_PERM_ENCRYPT    ? 'e' : '-'),
            (perm & SDC_PERM_DECRYPT    ? 'd' : '-'),
            (perm & SDC_PERM_SIGN       ? 's' : '-'),
            (perm & SDC_PERM_VERIFY     ? 'v' : '-'),
            (perm & SDC_PERM_WRAP       ? 'w' : '-'),
            (perm & SDC_PERM_UNWRAP     ? 'u' : '-'),
            (perm & SDC_PERM_WRAP_KEY   ? 'k' : '-'),
            (perm & SDC_PERM_UNWRAP_KEY ? 'K' : '-'),
            (perm & SDC_PERM_DELETE     ? 'D' : '-'), space);
}

sdc_error_t sdc_permissions_write(const sdc_permissions_t *perms,
                                  char *buffer, size_t len)
{
    if (!perms || !buffer)
        return SDC_INVALID_PARAMETER;

    /* Six entries, each nine characters */
    if (len < (6 * 10))
        return SDC_INVALID_PARAMETER;

    _write_perms(perms->perms_owner,          buffer +  0, " ");
    _write_perms(perms->perms_group,          buffer + 10, " ");
    _write_perms(perms->perms_others,         buffer + 20, " ");
    _write_perms(perms->perms_inherit_owner,  buffer + 30, " ");
    _write_perms(perms->perms_inherit_group,  buffer + 40, " ");
    _write_perms(perms->perms_inherit_others, buffer + 50, "" );
    return SDC_OK;
}

/** Setters and getters **/

sdc_error_t sdc_permissions_owner_set(sdc_permissions_t *permissions, sdc_perm_bmsk_t bmsk)
{
    if (!permissions)
        return SDC_INVALID_PARAMETER;

    if (bmsk &
        ~(SDC_PERM_UID_MAX))
        return SDC_PERM_INVALID;

    permissions->perms_owner = bmsk;

    return SDC_OK;
}

sdc_error_t sdc_permissions_owner_get(const sdc_permissions_t *permissions, sdc_perm_bmsk_t *bmsk)
{
    if ((!permissions) || (!bmsk))
        return SDC_INVALID_PARAMETER;

    *bmsk = permissions->perms_owner;

    return SDC_OK;
}

sdc_error_t sdc_permissions_inherit_owner_set(sdc_permissions_t *permissions, sdc_perm_bmsk_t bmsk)
{
    if (!permissions)
        return SDC_INVALID_PARAMETER;

    if (bmsk &
        ~(SDC_PERM_INHERIT_UID_MAX))
        return SDC_PERM_INVALID;

    permissions->perms_inherit_owner = bmsk;

    return SDC_OK;
}

sdc_error_t sdc_permissions_inherit_owner_get(const sdc_permissions_t *permissions, sdc_perm_bmsk_t *bmsk)
{
    if ((!permissions) || (!bmsk))
        return SDC_INVALID_PARAMETER;

    *bmsk = permissions->perms_inherit_owner;

    return SDC_OK;
}

sdc_error_t sdc_permissions_group_set(sdc_permissions_t *permissions, sdc_perm_bmsk_t bmsk)
{
    if (!permissions)
        return SDC_INVALID_PARAMETER;

    if (bmsk &
        ~(SDC_PERM_GID_MAX))
        return SDC_PERM_INVALID;

    permissions->perms_group = bmsk;

    return SDC_OK;
}

sdc_error_t sdc_permissions_group_get(const sdc_permissions_t *permissions, sdc_perm_bmsk_t *bmsk)
{
    if ((!permissions) || (!bmsk))
        return SDC_INVALID_PARAMETER;

    *bmsk = permissions->perms_group;

    return SDC_OK;
}

sdc_error_t sdc_permissions_inherit_group_set(sdc_permissions_t *permissions, sdc_perm_bmsk_t bmsk)
{
    if (!permissions)
        return SDC_INVALID_PARAMETER;

    if (bmsk &
        ~(SDC_PERM_INHERIT_GID_MAX))
        return SDC_PERM_INVALID;

    permissions->perms_inherit_group = bmsk;

    return SDC_OK;
}

sdc_error_t sdc_permissions_inherit_group_get(const sdc_permissions_t *permissions, sdc_perm_bmsk_t *bmsk)
{
    if ((!permissions) || (!bmsk))
        return SDC_INVALID_PARAMETER;

    *bmsk = permissions->perms_inherit_group;

    return SDC_OK;
}

sdc_error_t sdc_permissions_others_set(sdc_permissions_t *permissions, sdc_perm_bmsk_t bmsk)
{
    if (!permissions)
        return SDC_INVALID_PARAMETER;

    if (bmsk &
        ~(SDC_PERM_OTHERS_MAX))
        return SDC_PERM_INVALID;

    permissions->perms_others = bmsk;

    return SDC_OK;
}

sdc_error_t sdc_permissions_others_get(const sdc_permissions_t *permissions, sdc_perm_bmsk_t *bmsk)
{
    if ((!permissions) || (!bmsk))
        return SDC_INVALID_PARAMETER;

    *bmsk = permissions->perms_others;

    return SDC_OK;
}

sdc_error_t sdc_permissions_inherit_others_set(sdc_permissions_t *permissions, sdc_perm_bmsk_t bmsk)
{
    if (!permissions)
        return SDC_INVALID_PARAMETER;

    if (bmsk &
        ~(SDC_PERM_INHERIT_OTHERS_MAX))
        return SDC_PERM_INVALID;

    permissions->perms_inherit_others = bmsk;

    return SDC_OK;
}

sdc_error_t sdc_permissions_inherit_others_get(const sdc_permissions_t *permissions, sdc_perm_bmsk_t *bmsk)
{
    if ((!permissions) || (!bmsk))
        return SDC_INVALID_PARAMETER;

    *bmsk = permissions->perms_inherit_others;

    return SDC_OK;
}
